// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit
import MSSOrchestration

class RegistrationViewController: UIViewController {
    @IBOutlet weak var content: UIScrollView!
    
    @IBOutlet private weak var userIdLbl: UILabel!
    @IBOutlet private weak var userId: UITextField!
    @IBOutlet private weak var userIdError: UILabel!
    @IBOutlet weak var activationPasswordLbl: UILabel!
    @IBOutlet weak var activationPassword: UITextField!
    @IBOutlet weak var activationPasswordError: UILabel!
    @IBOutlet weak var registerButton: UIButton!
    
    private weak var responder: UIView?
    private var progressDialog: UIView?
    private var orchestrator: Orchestrator?
    private let orchestrationDelegate = OrchestrationSampleDelegate()
    private var userIdToStore = ""
    private var serverCallHandler = ServerCallHandler()
    private let userAuthenticationDelegate = UserAuthenticationViewController()
    
    var hidesBackButton = false
    
    override func viewDidLoad() {
        super.viewDidLoad()
        setupOrchestration()
        setupView()
    }
    
    // MARK: Setup
    private func setupView() {
        userIdLbl.text = "prompt_user_id".localized
        activationPasswordLbl.text = "prompt_activation_password".localized
        registerButton.setTitle("btn_register".localized.uppercased(), for: .normal)
        
        navigationItem.setHidesBackButton(hidesBackButton, animated: true)
        title = "title_activity_registration".localized
        
        content.addGestureRecognizer(UITapGestureRecognizer(target: self, action: #selector(closeKeyboard)))
        
        registerButton.layer.cornerRadius = 3
    }
    
    private func setupOrchestration() {
        orchestrationDelegate.progressDialog = progressDialog
        orchestrationDelegate.viewController = self
        orchestrator = OrchestratorUtils.getOrchestrator(delegate: orchestrationDelegate)
        
        serverCallHandler.viewController = self
        serverCallHandler.orchestrator = orchestrator
    }
    
    // MARK: IBAction
    
    @IBAction func startRegistration(_ sender: Any) {
        view.endEditing(true)
        
        // Get input strings
        let activationPasswordStr: String
        if let userId = userId.text,
           let activationPassword = activationPassword.text {
            userIdToStore = userId
            activationPasswordStr = activationPassword
        } else {
            activationPasswordStr = ""
            assertionFailure("can't get userId or activation password")
        }
        
        // Reset errors
        userIdError.text = ""
        activationPasswordError.text = ""
        
        // Start progress dialogs
        progressDialog = UIUtils.displayProgress(controller: self, message: "dialog_progress_registering".localized)
        orchestrationDelegate.progressDialog = progressDialog
        
        // Create activation configuration
        let orchestrationUser = OrchestrationUser(identifier: userIdToStore, domain: nil)
        let activationParams = OnlineActivationParameters(user: orchestrationUser, activationPassword: activationPasswordStr, delegate: self)
        
        // Used for custom password instead of default one
        if let orchestrator = orchestrator {
            userAuthenticationDelegate.useExternalPassword(orchestrator: orchestrator, viewController: self)
        } else {
            assertionFailure("orchestrator is nil in RegistrationVC")
        }
        
        // Start activation
        orchestrator?.startActivation(with: activationParams)
    }
    
    private func sendCommandToServer(_ command: String) {
        serverCallHandler.progressDialog = progressDialog
        serverCallHandler.sendCommandToServer(command)
    }
    
    private func displayAlert(title: String, message: String, dismissAction: (() -> Void)? = nil) {
        var controller: UIViewController? = self
        
        while controller?.presentingViewController != nil {
            controller = controller?.presentingViewController
        }
        
        if let controller = controller {
            UIUtils.displayAlert(controller: controller, title: title, message: message, dismissAction: dismissAction)
        }
    }
    
    private func displayProgress(message: String) {
        if progressDialog != nil {
            UIUtils.hideProgress(progressDialog)
        }
        
        progressDialog = UIUtils.displayProgress(controller: self, message: message)
        orchestrationDelegate.progressDialog = progressDialog
    }
    
    private func backToMain() {
        navigationController?.popViewController(animated: true)
    }
    
    @objc private func closeKeyboard() {
        responder?.resignFirstResponder()
        responder = nil
    }
}

extension RegistrationViewController: UITextFieldDelegate {
    func textFieldDidBeginEditing(_ textField: UITextField) {
        responder = textField
    }
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        
        if textField == userId {
            activationPassword.becomeFirstResponder()
        }
        
        return true
    }
}

extension RegistrationViewController: OnlineActivationDelegate {
    func orchestrator(_ orchestrator: Orchestrator, didCompleteActivationStepWith command: String) {
        // Display progress dialog and send command
        UIUtils.hideProgress(progressDialog)
        progressDialog = UIUtils.displayProgress(controller: self, message: "dialog_progress_registering".localized)
        orchestrationDelegate.progressDialog = progressDialog
        sendCommandToServer(command)
    }
    
    func orchestratorDidFinishActivationWithSuccess(_ orchestrator: Orchestrator) {
        // Hide progress
        UIUtils.hideProgress(progressDialog)
        
        // Store activated user identity in the local preferences
        SharedPreferenceStorage.storeActivatedUser(activatedUser: userIdToStore)
        
        // Back to calling activity
        backToMain()
    }
    
    func orchestratorDidAbortActivation(_ orchestrator: Orchestrator) {
        // Hide progress
        UIUtils.hideProgress(progressDialog)
        UIUtils.displayAlert(controller: self, title: "dialog_title_activation".localized, message: "dialog_content_activation_abortion".localized)
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveActivationError error: OrchestrationError) {
        // Hide progress
        UIUtils.hideProgress(progressDialog)
        
        // User got it wrong, display specific error
        let errorText = [
            error.errorDescription,
            "code: \(error.errorCode)"
        ].compactMap { $0 }.joined(separator: " - ")
        
        switch error {
        case .userIDNullOrEmpty, .userIDWrongFormat:
            userIdError.text = errorText
            
        case .activationPasswordNullOrEmpty, .activationPasswordWrongLength, .activationPasswordWrongChecksum:
            activationPasswordError.text = errorText
            
        default:
            UIUtils.displayAlert(for: error, on: self)
        }
    }
}
